<?php

namespace App\Modules\Projects\Controllers;

use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\File;
use App\Modules\Projects\Models\Projects;
use App\Modules\Projects\Requests\SortRequest;
use App\Modules\Projects\Requests\StoreRequest;
use App\Modules\Projects\Requests\UpdateRequest;
use App\Modules\Projects\Resources\ProjectsResource;

class adminController extends Controller
{
    public function index()
    {
        $projects = Projects::with('Card', 'Links', 'Data')->get();
        return $this->sendResponse(ProjectsResource::collection($projects), 'Projects');
    }

    public function store(StoreRequest $request)
    {
        try {
            DB::beginTransaction();
            $images = [];
            foreach ($request->project_images as $image) {
                $images[] = $this->moveImage($image, 'projects');
            }

            $project = Projects::create([
                'images' => json_encode($images),
                'sort' => $this->getLatestNumber(),
            ]);
            $project->Data()->create([
                'title' => $request->project_title,
                'content' => json_encode($request->project_content),
            ]);

            $project->Links()->createMany($request->project_links);

            $project->Card()->create([
                'image' => $this->moveImage($request->card_image, 'projects'),
            ]);

            $card = $project->Card;
            if ($card) {
                $card->Data()->create([
                    'title' => $request->card_title,
                    'description' => $request->card_description,
                ]);
            }

            DB::commit();
            return $this->sendResponse(new ProjectsResource($project), 'Project Created');
        } catch (\Exception $e) {
            dd($e->getMessage());
            DB::rollBack();
            return $this->sendError($e->getMessage());
        }
    }

    public function edit($id)
    {
        $project = Projects::with('Card', 'Links', 'Data')->find($id);
        if (!$project) {
            return $this->sendError('Project Not Found');
        }
        return $this->sendResponse(new ProjectsResource($project), 'Project');
    }

    public function show($id)
    {
        $project = Projects::with('Card', 'Links', 'Data')->find($id);
        if (!$project) {
            return $this->sendError('Project Not Found');
        }
        return $this->sendResponse(new ProjectsResource($project), 'Project');
    }

    public function update(UpdateRequest $request, $id)
    {
        $project = Projects::with('Card', 'Links', 'Data')->find($id);
        if (!$project) {
            return $this->sendError('Project Not Found');
        }
        $project->Data()->update([
            'title' => $request->project_title,
            'content' => json_encode($request->project_content),
        ]);
        $project->Links()->delete();
        $project->Links()->createMany($request->project_links);
        $project->Card()->update([
            'image' => $this->updateImage($request->card_image, 'projects'),
        ]);
        $card = $project->Card;
        if ($card) {
            $card->Data()->update([
                'title' => $request->card_title,
                'description' => $request->card_description,
            ]);
        }
        return $this->sendResponse(new ProjectsResource($project), 'Project Updated');
    }

    private function moveImage($image_name, $objectId)
    {
        $imagePath = public_path('temp/' . $image_name);

        // Define the new directory structure
        $uploadsDir = public_path('uploads/' . $objectId);

        // Create the directories if they do not exist
        if (!File::exists($uploadsDir)) {
            File::makeDirectory($uploadsDir, 0755, true);
        }

        // Move the image to the new directory
        $newImagePath = $uploadsDir . '/' . $image_name;
        File::move($imagePath, $newImagePath);

        $storedImagePath = 'uploads/' . $objectId . '/' . $image_name;
        return $storedImagePath;
    }

    private function updateImage($image_name, $objectId)
    {
        $imagePath = public_path('temp/' . $image_name);
        if (File::exists($imagePath)) {
            return $this->moveImage($image_name, $objectId);
        }
        return $image_name;
    }

    // delete project
    public function destroy($id)
    {
        $project = Projects::find($id);
        if (!$project) {
            return $this->sendError('Project Not Found');
        }
        $project->delete();
        return $this->sendResponse([], 'Project Deleted');
    }

    public function sortProjects(SortRequest $request)
    {
        DB::transaction(function () use ($request) {
            // Step 1: Fetch the target project
            $project = Projects::find($request->project_id);

            if (!$project) {
                throw new \Exception("Project not found.");
            }

            // Step 2: Temporarily remove the current project's sort
            $oldSort = $project->sort;
            $project->update(['sort' => null]); // Temporarily set to null to avoid conflicts

            // Step 3: Shift other projects' sort values
            if ($oldSort < $request->sort) {
                // Moving down: Increment sort for projects between old and new sort positions
                Projects::where('id', '!=', $request->project_id)
                    ->whereBetween('sort', [$oldSort + 1, $request->sort])
                    ->decrement('sort');
            } else {
                // Moving up: Increment sort for projects between new and old sort positions
                Projects::where('id', '!=', $request->project_id)
                    ->whereBetween('sort', [$request->sort, $oldSort - 1])
                    ->increment('sort');
            }

            // Step 4: Assign the new sort value to the updated project
            $project->update(['sort' => $request->sort]);

            // Step 5: Reset all sort values sequentially to avoid gaps
            $allProjects = Projects::orderBy('sort')->get();
            $counter = 1;

            foreach ($allProjects as $proj) {
                $proj->update(['sort' => $counter]);
                $counter++;
            }
        });

        return $this->sendResponse([], 'Projects sorted successfully');
    }


    protected function getLatestNumber()
    {
        $faq = Projects::orderBy('sort', 'desc');
        $faq->limit(1);
        $data = $faq->first() ?? (object)['sort' => 0];
        return $data->sort + 1;
    }
}
