import { Button, Group } from "@mantine/core";
import { useTranslation } from "react-i18next";
import { useFormik } from "formik";
import * as Yup from "yup";
import axios from "axios";
import Swal from 'sweetalert2';
import { useState } from "react";

const ContactUs = () => {
  const { t, i18n } = useTranslation();
  const currentLanguage = i18n.language;
  const textDirection = currentLanguage === "ar" ? "rtl" : "ltr";

  const [isSubmitting, setIsSubmitting] = useState(false);


  // Define validation schema with Yup
  const validationSchema = Yup.object({
    name: Yup.string().required(t("contact.form.nameRequired")),
    email: Yup.string().email(t("contact.form.invalidEmail")).required(t("contact.form.emailRequired")),
    phone: Yup.string().matches(/^[0-9]{10,15}$/, t("contact.form.invalidPhone")).required(t("contact.form.numberRequired")),
    subject: Yup.string().required(t("contact.form.subjectRequired")),
  });

  const formik = useFormik({
    initialValues: {
      name: "",
      email: "",
      message: "",
      phone: "",
      subject: "",
    },
    validationSchema,
    onSubmit: async (values) => {
      setIsSubmitting(true);
      try {
        const response = await axios.post("https://back-soft.softlabprog.cloud/api/user/contacts", values);
        if (response.status === 200) {
          Swal.fire({
            title: 'Success!',
            text: t('contact.form.alert.success') ,
            icon: 'success',
            confirmButtonText: 'OK'
          });
          formik.resetForm();
        }
      }
      catch (error) {
        Swal.fire({
          title: 'Error!',
          text: t('contact.form.alert.error'),
          icon: 'error',
          confirmButtonText: 'OK'
        });
      }
      finally {
        setIsSubmitting(false);
      }
    },
  });

  return (
    <section className="lg:py-12 py-6" id="contact-us">
      <div className={`grid grid-cols-1 lg:grid-cols-2 container mx-auto p-5 lg:pr-16 bg-[#F5F5F5] rounded-2xl`}>
        <div className="contact-us-bg lg:mr-10 py-32 lg:py-48 flex justify-center flex-col relative">
          <div className="flex flex-col">
          <div className="lg:ml-10 ml-5 space-y-8 lg:space-y-16">
              <div className="flex items-start gap-5">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M12 1.5C9.81276 1.50258 7.71584 2.3726 6.16922 3.91922C4.6226 5.46584 3.75258 7.56276 3.75 9.75C3.74738 11.5374 4.33124 13.2763 5.412 14.7C5.412 14.7 5.637 14.9963 5.67375 15.039L12 22.5L18.3293 15.0353C18.3623 14.9955 18.588 14.7 18.588 14.7L18.5888 14.6978C19.669 13.2747 20.2526 11.5366 20.25 9.75C20.2474 7.56276 19.3774 5.46584 17.8308 3.91922C16.2842 2.3726 14.1872 1.50258 12 1.5ZM12 12.75C11.4067 12.75 10.8266 12.5741 10.3333 12.2444C9.83994 11.9148 9.45543 11.4462 9.22836 10.8981C9.0013 10.3499 8.94189 9.74667 9.05765 9.16473C9.1734 8.58279 9.45912 8.04824 9.87868 7.62868C10.2982 7.20912 10.8328 6.9234 11.4147 6.80764C11.9967 6.69189 12.5999 6.7513 13.1481 6.97836C13.6962 7.20542 14.1648 7.58994 14.4944 8.08329C14.8241 8.57664 15 9.15666 15 9.75C14.999 10.5453 14.6826 11.3078 14.1202 11.8702C13.5578 12.4326 12.7953 12.749 12 12.75Z" fill="white" />
                </svg>
                <p className="font-sfpro text-white text-base">+1012 3456 789</p>
              </div>
              <div className="flex items-center gap-5">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M22 4H2V20H22V4ZM20 8L12 13L4 8V6L12 11L20 6V8Z" fill="white" />
                </svg>
                <p className="font-sfpro text-white text-base">demo@gmail.com</p>
              </div>
              <div className="flex items-center gap-5">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M12 1.5C9.81276 1.50258 7.71584 2.3726 6.16922 3.91922C4.6226 5.46584 3.75258 7.56276 3.75 9.75C3.74738 11.5374 4.33124 13.2763 5.412 14.7C5.412 14.7 5.637 14.9963 5.67375 15.039L12 22.5L18.3293 15.0353C18.3623 14.9955 18.588 14.7 18.588 14.7L18.5888 14.6978C19.669 13.2747 20.2526 11.5366 20.25 9.75C20.2474 7.56276 19.3774 5.46584 17.8308 3.91922C16.2842 2.3726 14.1872 1.50258 12 1.5ZM12 12.75C11.4067 12.75 10.8266 12.5741 10.3333 12.2444C9.83994 11.9148 9.45543 11.4462 9.22836 10.8981C9.0013 10.3499 8.94189 9.74667 9.05765 9.16473C9.1734 8.58279 9.45912 8.04824 9.87868 7.62868C10.2982 7.20912 10.8328 6.9234 11.4147 6.80764C11.9967 6.69189 12.5999 6.7513 13.1481 6.97836C13.6962 7.20542 14.1648 7.58994 14.4944 8.08329C14.8241 8.57664 15 9.15666 15 9.75C14.999 10.5453 14.6826 11.3078 14.1202 11.8702C13.5578 12.4326 12.7953 12.749 12 12.75Z" fill="white" />
                </svg>
                <p className="font-sfpro lg:w-1/2 text-white text-xs lg:text-base">132 Dartmouth Street Boston, Massachusetts 02156 United States</p>
              </div>
            </div>
          </div>
          <div className="flex items-end ml-10 gap-5 absolute left-12 lg:left-0 bottom-10">
          <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <g clipPath="url(#clip0_612_39613)">
                <path fillRule="evenodd" clipRule="evenodd" d="M12 0C5.3724 0 0 5.3808 0 12.0204C0 17.3304 3.438 21.8364 8.2068 23.4252C8.8068 23.5356 9.0252 23.1648 9.0252 22.8456C9.0252 22.5612 9.0156 21.804 9.0096 20.802C5.6712 21.528 4.9668 19.1904 4.9668 19.1904C4.422 17.8008 3.6348 17.4312 3.6348 17.4312C2.5452 16.6872 3.7176 16.7016 3.7176 16.7016C4.9212 16.7856 5.5548 17.94 5.5548 17.94C6.6252 19.776 8.364 19.2456 9.0468 18.9384C9.1572 18.162 9.4668 17.6328 9.81 17.3328C7.146 17.0292 4.344 15.9972 4.344 11.3916C4.344 10.08 4.812 9.006 5.5788 8.166C5.4552 7.8624 5.0436 6.6396 5.6964 4.986C5.6964 4.986 6.7044 4.662 8.9964 6.2172C9.97532 5.95022 10.9853 5.81423 12 5.8128C13.02 5.8176 14.046 5.9508 15.0048 6.2172C17.2956 4.662 18.3012 4.9848 18.3012 4.9848C18.9564 6.6396 18.5436 7.8624 18.4212 8.166C19.1892 9.006 19.6548 10.08 19.6548 11.3916C19.6548 16.0092 16.848 17.0256 14.1756 17.3232C14.6064 17.694 14.9892 18.4272 14.9892 19.5492C14.9892 21.1548 14.9748 22.452 14.9748 22.8456C14.9748 23.1672 15.1908 23.5416 15.8004 23.424C18.19 22.6225 20.2672 21.0904 21.7386 19.0441C23.2099 16.9977 24.001 14.5408 24 12.0204C24 5.3808 18.6264 0 12 0Z" fill="white" />
              </g>
              <defs>
                <clipPath id="clip0_612_39613">
                  <rect width="24" height="24" fill="white" />
                </clipPath>
              </defs>
            </svg>
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path fillRule="evenodd" clipRule="evenodd" d="M15.9455 23L10.396 15.0901L3.44886 23H0.509766L9.09209 13.2311L0.509766 1H8.05571L13.286 8.45502L19.8393 1H22.7784L14.5943 10.3165L23.4914 23H15.9455ZM19.2185 20.77H17.2398L4.71811 3.23H6.6971L11.7121 10.2532L12.5793 11.4719L19.2185 20.77Z" fill="white" />
            </svg>
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <g clipPath="url(#clip0_612_39611)">
                <path d="M22.2283 0H1.77167C1.30179 0 0.851161 0.186657 0.518909 0.518909C0.186657 0.851161 0 1.30179 0 1.77167V22.2283C0 22.6982 0.186657 23.1488 0.518909 23.4811C0.851161 23.8133 1.30179 24 1.77167 24H22.2283C22.6982 24 23.1488 23.8133 23.4811 23.4811C23.8133 23.1488 24 22.6982 24 22.2283V1.77167C24 1.30179 23.8133 0.851161 23.4811 0.518909C23.1488 0.186657 22.6982 0 22.2283 0ZM7.15333 20.445H3.545V8.98333H7.15333V20.445ZM5.34667 7.395C4.93736 7.3927 4.53792 7.2692 4.19873 7.04009C3.85955 6.81098 3.59584 6.48653 3.44088 6.10769C3.28591 5.72885 3.24665 5.31259 3.32803 4.91145C3.40941 4.51032 3.6078 4.14228 3.89816 3.85378C4.18851 3.56529 4.55782 3.36927 4.95947 3.29046C5.36112 3.21165 5.77711 3.25359 6.15495 3.41099C6.53279 3.56838 6.85554 3.83417 7.08247 4.17481C7.30939 4.51546 7.43032 4.91569 7.43 5.325C7.43386 5.59903 7.38251 5.87104 7.27901 6.1248C7.17551 6.37857 7.02198 6.6089 6.82757 6.80207C6.63316 6.99523 6.40185 7.14728 6.14742 7.24915C5.893 7.35102 5.62067 7.40062 5.34667 7.395ZM20.4533 20.455H16.8467V14.1933C16.8467 12.3467 16.0617 11.7767 15.0483 11.7767C13.9783 11.7767 12.9283 12.5833 12.9283 14.24V20.455H9.32V8.99167H12.79V10.58H12.8367C13.185 9.875 14.405 8.67 16.2667 8.67C18.28 8.67 20.455 9.865 20.455 13.365L20.4533 20.455Z" fill="white" />
              </g>
              <defs>
                <clipPath id="clip0_612_39611">
                  <rect width="24" height="24" fill="white" />
                </clipPath>
              </defs>
            </svg>
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <g clipPath="url(#clip0_612_39612)">
                <path d="M24 12C24 5.37258 18.6274 0 12 0C5.37258 0 0 5.37258 0 12C0 17.9895 4.3882 22.954 10.125 23.8542V15.4688H7.07812V12H10.125V9.35625C10.125 6.34875 11.9166 4.6875 14.6576 4.6875C15.9701 4.6875 17.3438 4.92188 17.3438 4.92188V7.875H15.8306C14.34 7.875 13.875 8.80008 13.875 9.75V12H17.2031L16.6711 15.4688H13.875V23.8542C19.6118 22.954 24 17.9895 24 12Z" fill="white" />
              </g>
              <defs>
                <clipPath id="clip0_612_39612">
                  <rect width="24" height="24" fill="white" />
                </clipPath>
              </defs>
            </svg>
          </div>
        </div>
        <div className="mt-8 w-full">
          <form onSubmit={formik.handleSubmit} style={{ direction: textDirection }}>
            <div className="grid grid-cols-1 lg:grid-cols-2 w-full mt-8 mb-7 lg:mb-14 gap-7">
              <div className="w-full">
                <label className="block mb-2 text-[10px] md:text-lg">{t("contact.form.name")}</label>
                <input
                  type="text"
                  id="name"
                  {...formik.getFieldProps("name")}
                  className="w-full p-2 border border-gray-300 rounded input-style bg-[#F5F5F5]"
                  style={{ direction: textDirection, textAlign: textDirection === "rtl" ? "right" : "left" }}
                />
                {formik.touched.name && formik.errors.name ? (
                  <div className="text-red-500 text-xs">{formik.errors.name}</div>
                ) : null}
              </div>
              <div className="w-full">
                <label className="block mb-2 text-[10px] md:text-lg">{t("contact.form.number")}</label>
                <input
                  type="tel"
                  id="phone"
                  {...formik.getFieldProps("phone")}
                  className="w-full p-2 border border-gray-300 rounded input-style bg-[#F5F5F5]"
                  style={{ direction: textDirection, textAlign: textDirection === "rtl" ? "right" : "left" }}
                />
                {formik.touched.phone && formik.errors.phone ? (
                  <div className="text-red-500 text-xs">{formik.errors.phone}</div>
                ) : null}
              </div>
            </div>
            <div className="grid grid-cols-1 lg:grid-cols-2 w-full mt-8 mb-6 lg:mb-14 gap-7">
              <div className="w-full">
                <label className="block mb-2 text-[10px] md:text-lg">{t("contact.form.subject")}</label>
                <input
                  type="text"
                  id="subject"
                  {...formik.getFieldProps("subject")}
                  className="w-full p-2 border border-gray-300 rounded input-style bg-[#F5F5F5]"
                  style={{ direction: textDirection, textAlign: textDirection === "rtl" ? "right" : "left" }}
                />
                {formik.touched.subject && formik.errors.subject ? (
                  <div className="text-red-500 text-xs">{formik.errors.subject}</div>
                ) : null}
              </div>
              <div className="w-full">
                <label className="block mb-2 text-[10px] md:text-lg">{t("contact.form.email")}</label>
                <input
                  type="email"
                  id="email"
                  {...formik.getFieldProps("email")}
                  className="w-full p-2 border border-gray-300 rounded input-style bg-[#F5F5F5]"
                  style={{ direction: textDirection, textAlign: textDirection === "rtl" ? "right" : "left" }}
                />
                {formik.touched.email && formik.errors.email ? (
                  <div className="text-red-500 text-xs">{formik.errors.email}</div>
                ) : null}
              </div>
            </div>
            <div className="w-full mt-8">
              <label className="block mb-2 text-[10px] md:text-lg">{t("contact.form.message")}</label>
              <textarea
                id="message"
                {...formik.getFieldProps("message")}
                className="w-full p-2 mt-5 border border-gray-300 rounded input-style bg-[#F5F5F5]"
                rows={2}
                style={{ direction: textDirection, textAlign: textDirection === "rtl" ? "right" : "left" }}
              />
              {formik.touched.message && formik.errors.message ? (
                <div className="text-red-500 text-xs">{formik.errors.message}</div>
              ) : null}
            </div>
            <Group mt="md" className="flex">
              <Button
                loading={isSubmitting}
                type="submit"
                className="font-laxr bg-gradient-to-r from-[#0C3580] to-[#1BBBCF] border-none rounded-lg lg:w-60 lg:h-16 w-36 h-11 shadow-none mt-5 lg:mt-14"
              >
                <span>
                  <svg
                    width="25"
                    height="24"
                    viewBox="0 0 25 24"
                    fill="none"
                    xmlns="http://www.w3.org/2000/svg"
                  >
                    <path
                      d="M5.5 12H19.5"
                      stroke="white"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                    <path
                      d="M12.5 5L19.5 12L12.5 19"
                      stroke="white"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                  </svg>
                </span>
                {t("contact.form.button")}
              </Button>
            </Group>
          </form>
        </div>
      </div>
    </section>
  );
};

export default ContactUs;